from flask import Flask, render_template, request
import os
import pandas as pd

app = Flask(__name__)

UPLOAD_FOLDER = "uploads"
app.config["UPLOAD_FOLDER"] = UPLOAD_FOLDER

if not os.path.exists(UPLOAD_FOLDER):
    os.makedirs(UPLOAD_FOLDER)


# ==========================
# ANALISA OTOMATIS EMA 20/50
# ==========================
def analyze_xauusd(price_history):

    df = pd.DataFrame(price_history, columns=["close"])

    df["ema20"] = df["close"].ewm(span=20, adjust=False).mean()
    df["ema50"] = df["close"].ewm(span=50, adjust=False).mean()

    last = df.iloc[-1]

    price = last["close"]
    ema20 = last["ema20"]
    ema50 = last["ema50"]

    risk = 5.0
    reward = 10.0

    # ================= TREND DETECTION =================
    if ema20 > ema50:
        trend = "BULLISH"
        signal = "BUY"
        sl = price - risk
        tp = price + reward
    elif ema20 < ema50:
        trend = "BEARISH"
        signal = "SELL"
        sl = price + risk
        tp = price - reward
    else:
        trend = "SIDEWAYS"
        signal = "NO TRADE"
        sl = None
        tp = None

    # ================= CONFIDENCE =================
    distance = abs(ema20 - ema50)

    if distance > 3:
        confidence = 90
    elif distance > 2:
        confidence = 82
    elif distance > 1:
        confidence = 74
    else:
        confidence = 60

    return {
        "trend": trend,
        "signal": signal,
        "entry": round(price, 2),
        "sl": round(sl, 2) if sl else None,
        "tp": round(tp, 2) if tp else None,
        "sl_pips": 50 if sl else None,
        "tp_pips": 100 if tp else None,
        "confidence": f"{confidence}%"
    }


# ==========================
# ROUTE
# ==========================
@app.route("/", methods=["GET", "POST"])
def index():
    result = None
    filename = None

    if request.method == "POST":
        try:
            price = float(request.form["price"])

            # Simulasi 60 harga terakhir (biar EMA valid)
            price_history = [price - i*0.3 for i in range(60)][::-1]

            file = request.files.get("chart")
            if file and file.filename != "":
                filepath = os.path.join(app.config["UPLOAD_FOLDER"], file.filename)
                file.save(filepath)
                filename = file.filename

            result = analyze_xauusd(price_history)

        except Exception as e:
            print("Error:", e)
            result = None

    return render_template("index.html", result=result, filename=filename)


if __name__ == "__main__":
    app.run(debug=True)